'''
    UI platform which demonstrates ui loading with PySide2,
    and displays an object tree explorer.
'''
import os
import sys
from PySide2.QtWidgets import QMessageBox
from PySide2 import QtCore
from PySide2.QtUiTools import QUiLoader

import pymxs

sys.path.append(os.path.dirname(os.path.abspath(__file__)))
from maxDataModels import maxTreeNode
from qtViewModels import vmodelMaxObjects

class PyMaxExplorer(QtCore.QObject):
    """
    Widget to demonstrate how the interaction between PySide2 and 3ds Max thru pymxs can work
    A tree view is built and updated as soon as a 3ds Max node is modified in the application
    """
    def __init__(self, parent=None):
        super(PyMaxExplorer, self).__init__(parent)

        ui_file = QtCore.QFile(os.path.dirname(os.path.realpath(__file__)) + "\\mainUI.ui")
        ui_file.open(QtCore.QFile.ReadOnly)
        self.window = QUiLoader().load(ui_file, parent)
        ui_file.close()

        self.build_view_model()
        self.connect_actions()
        self.setup_max_callbacks()

        self.window.show()

    def setup_max_callbacks(self):
        """ Setup 3ds Max node event callback """
        self._callback_item = pymxs.runtime.NodeEventCallback(all=self.callback_node_event)

    def teardown_max_callbacks(self):
        """ Remove registered 3ds Max callback """
        self._callback_item = None

    def build_view_model(self):
        """ Build view model of the scene graph """
        self._root_node = self.build_tree(pymxs.runtime.rootNode)
        self._proxy_model = QtCore.QSortFilterProxyModel()
        self._model = vmodelMaxObjects(self._root_node)
        self._proxy_model.setSourceModel(self._model)
        self._proxy_model.setDynamicSortFilter(True)
        self._proxy_model.setFilterCaseSensitivity(QtCore.Qt.CaseInsensitive)
        self.window.treeView_objects.setModel(self._proxy_model)
        self.window.treeView_objects.setSortingEnabled(True)
        QtCore.QObject.connect(
            self.window.uiGraphFilter,
            QtCore.SIGNAL("textChanged(QString)"),
            self._proxy_model.setFilterRegExp)

    def connect_actions(self):
        """ Connect action buttons """
        self.connect(self.window.actionWatch, QtCore.SIGNAL('triggered()'), self.action_watch)
        self.connect(self.window.actionAbout, QtCore.SIGNAL('triggered()'), self.action_about)
        self.connect(self.window.actionRefresh, QtCore.SIGNAL('triggered()'), self.action_refresh)
        self.connect(self.window.actionExit, QtCore.SIGNAL('triggered()'), self.action_exit)
        self.connect(self.window.actionUndo, QtCore.SIGNAL('triggered()'), self.action_undo)
        self.connect(self.window.actionRedo, QtCore.SIGNAL('triggered()'), self.action_redo)

    def action_watch(self):
        """ 'Watch' action script - raise Watcher widget """
        self.window.Watcher.show()

    def action_about(self):
        """ 'About' action script - display about box """
        QMessageBox.about(self.parent(),
                          "PyMaxExplorer 1.0",
                          "PyMaxExplorer is a demo for pymxs and PySide2 usage in 3ds Max")

    def action_refresh(self):
        """ 'Refresh' action script - trigger a refresh of the model view """
        self.build_view_model()

    def action_exit(self):
        """ 'Exit' action script - close UI """
        self.close()

    def action_undo(self):
        """ 'Undo' action script - undo last action done in 3ds Max """
        pymxs.run_undo()

    def action_redo(self):
        """ 'Redo' action script - redo last action done in 3ds Max """
        pymxs.run_redo()

    def callback_node_event(self, event, node):
        """ Method called on node event generated in 3ds Max """
        self.build_view_model()

    def build_tree(self, max_obj, parent=None, user_name=''):
        """ Build the node tree from a 3ds Max node """
        node_name = max_obj.name
        if user_name:
            node_name = user_name
        new_node = maxTreeNode(node_name, parent)
        for child in max_obj.children:
            self.build_tree(child, new_node)
        return new_node
